import kotlin.math.log2
import kotlin.math.min
import kotlin.math.pow

fun main() {
  val input = System.`in`.bufferedReader()
  val chipmunks = input.readLine().toInt()
  val data = (1..chipmunks).map {
    Action.fromString(input.readLine())
  }.toTypedArray().let { SegmentTree(it) }
  val groups = input.readLine().toInt()
  repeat(groups) {
    val (start, finish) = input.readLine().split(" ").map { it.toInt() }
    println(data.rangeMinQuery(start - 1, finish - 1).min())
  }
}

enum class Action {
  ROCK, PAPER, SCISSORS;

  companion object {
    fun fromString(line: String) = when (line) {
      "Paper" -> PAPER
      "Rock" -> ROCK
      "Scissors" -> SCISSORS
      else -> error("Invalid input")
    }
  }
}


class SegmentTree(array: Array<Action>) {
  data class Node(
    var from: Int, var to: Int, var paper: Int, var rock: Int, var scissors: Int
  ) {
    fun size() = to - from + 1

    operator fun plus(other: Node) = Node(0, 0, paper + other.paper, rock + other.rock, scissors + other.scissors)

    fun min() = min(rock, min(paper, scissors)) * 3
  }

  private val values: Array<Action> = array.copyOf()
  val size: Int = (2 * 2.0.pow(log2(array.size.toDouble()) + 1)).toInt()

  private val heap = Array(size) { Node(-1, -1, 0, 0, 0) }

  init {
    build(1, 0, array.size - 1)
  }

  private fun leftChild(pointer: Int) = pointer shl 1
  private fun rightChild(pointer: Int) = (pointer shl 1) + 1
  private fun contains(from: Int, to: Int, fromIn: Int, toIn: Int) = fromIn in from..to && toIn in from..to
  private fun intersects(from1: Int, to1: Int, from2: Int, to2: Int) = from2 in from1..to1 || from1 in from2..to2
  private fun build(pointer: Int, from: Int, to: Int) {
    val diff = to - from
    heap[pointer].from = from
    heap[pointer].to = to
    if (diff >= 1) {
      val m = (from + to) / 2
      build(leftChild(pointer), from, m)
      build(rightChild(pointer), m + 1, to)
    }
    heap[pointer].rock = when (diff) {
      0 -> if (values[from] == Action.ROCK) 1 else 0
      else -> heap[leftChild(pointer)].rock + heap[rightChild(pointer)].rock
    }
    heap[pointer].paper = when (diff) {
      0 -> if (values[from] == Action.PAPER) 1 else 0
      else -> heap[leftChild(pointer)].paper + heap[rightChild(pointer)].paper
    }
    heap[pointer].scissors = when (diff) {
      0 -> if (values[from] == Action.SCISSORS) 1 else 0
      else -> heap[leftChild(pointer)].scissors + heap[rightChild(pointer)].scissors
    }
  }


  fun rangeMinQuery(from: Int, to: Int) = rangeMinQuery(1, from, to)
  private fun rangeMinQuery(pointer: Int, from: Int, to: Int): Node {
    return when {
      contains(from, to, heap[pointer].from, heap[pointer].to) -> heap[pointer]
      intersects(from, to, heap[pointer].from, heap[pointer].to) ->
        rangeMinQuery(leftChild(pointer), from, to) + rangeMinQuery(rightChild(pointer), from, to)
      else -> Node(0, 0, 0, 0, 0)
    }
  }
}
