-- @EXPECTED_RESULTS@: CORRECT
import Data.List
import Control.Monad
import Text.Printf

-- Convex hull using Graham Scan from http://hynek.me/graham.hs
-- Adapted to use Integers
type P = (Integer, Integer)

cross :: P -> P -> P -> Integer
cross a b c = (fst b - fst a) * (snd c - snd a) - (fst c - fst a) * (snd b - snd a)

dist :: P -> P -> Double
dist (x1,y1) (x2,y2) = sqrt (fromInteger $ (x1-x2) * (x1-x2) + (y1-y2)*(y1-y2))

cmpCross :: P -> P -> P -> Ordering
cmpCross p a b = let cmp = compare (cross p a b) 0
                 in if cmp == EQ
                    then compare (dist p a) (dist p b)
                    else cmp

convex :: [P] -> [P]
convex xs = let pivot    = minimum xs
                sortedPs = sortBy (cmpCross pivot) (delete pivot (nub xs))
            in clean (tail sortedPs) [head sortedPs, pivot]

clean :: [P] -> [P] -> [P]
clean (x:xs) (a1:a2:acc) = if cross a2 a1 x > 0 && length acc > 0
                           then clean (x:xs) (a2:acc)
                           else clean xs (x:a1:a2:acc)
clean _ acc = acc

-- My code
pathLength :: [P] -> Double
pathLength (x:y:xs) = dist x y + pathLength (y:xs)
pathLength _        = 0

solve :: P -> P -> [P] -> Maybe Double
solve luke hatch daltons = do
  let hull = convex (luke:hatch:daltons)
  if luke `elem` hull && hatch `elem` hull
    then let p1 = pathLength $ (++[hatch]) $ takeWhile (/=hatch) $ dropWhile (/=luke) $ hull ++ hull
             p2 = pathLength $ (++[luke]) $ takeWhile (/=luke) $ dropWhile (/=hatch) $ hull ++ hull
         in  return $ p1 `min` p2
    else if (luke `elem` hull && hatch `elem` convex (hatch:daltons))
            || (hatch `elem` hull && luke `elem` convex (luke:daltons))
         then return $ dist luke hatch
         else Nothing

main :: IO ()
main = do c <- getLine
          replicateM_ (read c) $ do
            luke    <- getPoint
            escape  <- getPoint
            n       <- getLine
            daltons <- replicateM (read n) getPoint
            putStrLn $ maybe "IMPOSSIBLE" (printf "%.3f") $ solve luke escape daltons

getPoint :: IO P
getPoint = do ln <- getLine
              let [x,y] = map read $ words ln 
              return (x,y)