// @EXPECTED_RESULTS@: CORRECT
#include <assert.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

struct Shop
{
    int units_avail, unit_price, order_cost;
};

static int min(int x, int y) { return x < y ? x : y; }

/* Orders shops by unit_price desc, order-cost desc, units-avail asc. */
static int compare_unit_price(const void *a, const void *b)
{
    const struct Shop *s = a, *t = b;
    if (s->unit_price != t->unit_price) return s->unit_price - t->unit_price;
    if (s->order_cost != t->order_cost) return s->order_cost - t->order_cost;
    return t->units_avail - s->units_avail;
}

/* Orders shops by order-cost desc, unit_price desc, units-avail asc. */
static int compare_order_price(const void *a, const void *b)
{
    const struct Shop *s = a, *t = b;
    if (s->order_cost != t->order_cost) return s->order_cost - t->order_cost;
    if (s->unit_price != t->unit_price) return s->unit_price - t->unit_price;
    return t->units_avail - s->units_avail;
}

/* Orders shops by average price per unit (assuming we order the entire stock). */
static int compare_avg_price(const void *a, const void *b)
{
    const struct Shop *s = a, *t = b;
    double x = (double)(s->unit_price*s->units_avail + s->order_cost)/s->units_avail;
    double y = (double)(t->unit_price*t->units_avail + t->order_cost)/t->units_avail;
    if (x < y) return -1;
    if (x > y) return +1;
    return 0;
}

/* Orders shops primarily by a strict criterion (one shop is 'strictly better'
   than another if both its per-order and per-unit prices are no greater than
   that of the other, and at least one of them is different. */
static int compare_strict(const void *a, const void *b)
{
    const struct Shop *s = a, *t = b;
    int x = s->unit_price - t->unit_price,
        y = s->order_cost - t->order_cost;
    if (x <= 0 && y <= 0 && (x < 0 || y < 0)) return -1;
    if (x >= 0 && y >= 0 && (x > 0 || y > 0)) return +1;
    return compare_avg_price(a, b);  /* fall back to another criterion */
}

/* Solve greedily by first sorting shops by a fixed criterion (implemented by
   the `compare' function argument), buying from the first x shops, and then
   fulfilling the rest of the order from one of the remaining shops. */
static int solve_greedy_by( int (*compare)(const void *a, const void *b),
                            int total_units, int nshop, struct Shop shops[] )
{
    int best_price = INT_MAX, total_price = 0, i, j;
    qsort(shops, nshop, sizeof(struct Shop), compare);
    assert(total_units > 0);
    for (i = 0; i < nshop; ++i)
    {
        for (j = i; j < nshop; ++j)
        {
            if (total_units <= shops[j].units_avail)
            {
                best_price = min(best_price, total_price +
                    total_units*shops[j].unit_price + shops[j].order_cost);
            }
        }
        if (total_units < shops[i].units_avail) break;
        total_price += shops[i].order_cost;
        total_price += shops[i].units_avail * shops[i].unit_price;
        total_units -= shops[i].units_avail;
    }
    assert(best_price < INT_MAX);
    return best_price;
}

/* Combine a couple of greedy strategies. */
static int solve_greedy(int total_units, int nshop, struct Shop shops[])
{
    int a = solve_greedy_by(compare_unit_price,  total_units, nshop, shops),
        b = solve_greedy_by(compare_order_price, total_units, nshop, shops),
        c = solve_greedy_by(compare_avg_price,   total_units, nshop, shops),
        d = solve_greedy_by(compare_strict,      total_units, nshop, shops),
        res = min(min(a, b), min(c, d));
    /*
    fprintf( stderr, " %c%c%c%c %d %d %d %d => %d\n",
             " a"[a==res], " b"[b==res], " c"[c==res], " d"[d==res],
             a, b, c, d, res );
    */
    return res;
}

/* Sort shops by price, and then brute-force all possible subsets.
   This should yield the correct answer, but takes O(m * 2**m) time,
   so it only works for up to ~25 shops or so. */
static int solve_bruteforce(int total_units, int nshop, struct Shop shops[])
{
    int min_price = INT_MAX, mask;
    qsort(shops, nshop, sizeof(struct Shop), compare_unit_price);
    assert(nshop <= 30);
    for (mask = (1<<nshop) - 1; mask > 0; --mask)
    {
        int units = total_units, price = 0, i;
        assert(units > 0);
        for (i = 0; i < nshop; ++i)
        {
            if (mask & (1<<i))
            {
                price += shops[i].order_cost;
                if (units > shops[i].units_avail)
                {
                    price += shops[i].units_avail * shops[i].unit_price;
                    units -= shops[i].units_avail;
                }
                else
                {
                    price += units * shops[i].unit_price;
                    units = 0;
                }
            }
        }
        if (units == 0 && price < min_price) min_price = price;
    }
    assert(min_price > 0);
    return min_price;
}

/* Solve by dynamic programming in O(n*m) time and O(n) space.

   Approach:
    a) if we order from a set of i shops, then w.l.o.g. we can assume we buy
       the maximal number of units from the i-1 shops with the lowest per-unit
       prices (because otherwise, we can either move a product from the most
       expensive shop to a cheaper one, or eliminate the most expensive shop
       altogether)
    b) let memo[i][j] be the lowest price we can buy exactly j units at,
       assuming we use only the shops with the i lowest per-unit prices and
       at each such store we either buy all avaible units (at a cost of
       stock*unit_price + order_cost) or none at all.
    c) Now, for the j-th cheapest store, we can decide to buy k (0 < k <= stock)
       units and get the other n - k units from cheaper shops.  In other words,
       we are looking for the minimal value of k*unit_price[j] + order_cost[j] +
       memo[n - k][j].

    We can easily compute the memo (described at b) in O(n*m) time and then find
    the desired minimum (described in c) in time O(n*m).  To reduce the required
    memory to O(m), combine steps b and c and only maintain a single row of
    the memo.
*/
int solve_fast(int total_units, int nshop, struct Shop shops[])
{
    const int inf = 1000000000;
    int min_price, i, j, *memo;

    /* Sort shops by price. */
    qsort(shops, nshop, sizeof(struct Shop), compare_unit_price);

    /* Initialize memo. */
    memo = malloc(sizeof(*memo)*(total_units + 1));
    assert(memo != NULL);
    memo[0] = 0;
    for (i = 1; i <= total_units; ++i) memo[i] = inf;

    /* Process shops in order. */
    min_price = inf;
    for (i = 0; i < nshop; ++i)
    {
        /* Check if we can use this shop as the final one in a set of orders: */
        for (j = 0; j <= total_units && j <= shops[i].units_avail; ++j)
        {
            /* Buy j units from the i-th shop. */
            int price = memo[total_units - j] + shops[i].order_cost
                      + j * shops[i].unit_price;
            if (price < min_price) min_price = price;
        }

        /* Update prices if we buy all orders from this shop: */
        for (j = total_units; j >= shops[i].units_avail; --j)
        {
            int price = memo[j - shops[i].units_avail] + shops[i].order_cost
                      + shops[i].units_avail * shops[i].unit_price;
            if (price < memo[j]) memo[j] = price;
        }
        /* NB: the direction of the above loop is important -- if we increment
               j, we could try to use the same shop more than once! */
    }

    free(memo);
    return min_price;
}

/* TODO: add O(N*N*M) dynamic programming approach (which should be too slow) */

int main(int argc, char *argv[])
{
    int (*solve)(int, int, struct Shop*) = (void*)0; 
    int cases = 0, res;

    if (argc == 1) solve = solve_fast;

    if (argc == 2)
    {
        if (strcmp(argv[1], "greedy")     == 0) solve = solve_greedy; else
        if (strcmp(argv[1], "bruteforce") == 0) solve = solve_bruteforce; else
        if (strcmp(argv[1], "fast")       == 0) solve = solve_fast;
    }

    if (!solve)
    {
        printf("usage: %s [<method>]\n"
               "where method is `greedy', `bruteforce' or `fast'.\n", argv[0]);
        return 1;
    }

    res = scanf("%d", &cases);
    assert(res == 1);
    assert(0 < cases && cases <= 100);
    while (cases-- > 0)
    {
        int N = 0, M = 0, m;
        struct Shop S[100] = { { 0, 0, 0 } };
        res = scanf("%d %d", &N, &M);
        assert(res == 2);
        assert(1 <= N && N <= 10000);
        assert(1 <= M && M <= 100);
        for (m = 0; m < M; ++m)
        {
            res = scanf( "%d %d %d", &S[m].units_avail, &S[m].unit_price,
                                     &S[m].order_cost );
            assert(res == 3);
            assert(0 <= S[m].units_avail && S[m].units_avail <=   10000);
            assert(0 <= S[m].unit_price  && S[m].unit_price  <=   10000);
            assert(0 <= S[m].order_cost  && S[m].order_cost  <= 1000000);
        }
        printf("%d\n", solve(N, M, S));
    }
    return 0;
}
